<?php

namespace Modules;

use Flint\RenderComponent;

/**
 * Alert Component
 *
 * Displays color-coded alert boxes with icons for info, success, warning, and error messages.
 * Automatically includes appropriate styling and iconography based on alert type.
 */
class Alert extends RenderComponent
{
    /**
     * Render an alert box
     *
     * @param array  $props   Component properties (type: info|success|warning|error, title)
     * @param string $content Alert message content (pre-sanitized by Parser)
     * @return string Rendered HTML
     */
    public static function render(array $props, string $content): string
    {
        // RenderComponent::prop() reads props passed from markdown like {{Alert type="info"}}.
        $type = prop($props, 'type', 'info');
        // Optional title renders a bold heading inside the alert.
        $title = prop($props, 'title');

        $styles = [
            'info' => 'bg-blue-50 border-blue-200 text-blue-900',
            'success' => 'bg-green-50 border-green-200 text-green-900',
            'warning' => 'bg-yellow-50 border-yellow-200 text-yellow-900',
            'error' => 'bg-red-50 border-red-200 text-red-900',
        ];

        $icons = [
            'info' => '<svg class="w-5 h-5" fill="currentColor" viewBox="0 0 20 20"><path fill-rule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clip-rule="evenodd"/></svg>',
            'success' => '<svg class="w-5 h-5" fill="currentColor" viewBox="0 0 20 20"><path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"/></svg>',
            'warning' => '<svg class="w-5 h-5" fill="currentColor" viewBox="0 0 20 20"><path fill-rule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clip-rule="evenodd"/></svg>',
            'error' => '<svg class="w-5 h-5" fill="currentColor" viewBox="0 0 20 20"><path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clip-rule="evenodd"/></svg>',
        ];

        // Fall back to "info" if the type is unknown.
        $style = $styles[$type] ?? $styles['info'];
        $icon = $icons[$type] ?? $icons['info'];

        // The CMS parser already sanitized $content.
        $hasTitle = $title !== '';
        $hasContent = $content !== '';

        // Components return HTML strings, so we buffer a template-style output.
        ob_start();
        ?>
        <div class="motion-alert rounded-xl border p-4 my-4 <?= esc_html($style) ?>">
            <div class="flex gap-3">
                <!-- Icon is chosen by alert type. -->
                <div class="flex-shrink-0"><?= $icon ?></div>
                <div class="flex-1">
                    <!-- Title is optional to keep alerts compact. -->
                    <?php if ($hasTitle) : ?>
                        <h4 class="font-semibold mb-1"><?= esc_html($title) ?></h4>
                    <?php endif; ?>

                    <!-- Content is already parsed HTML from the CMS. -->
                    <?php if ($hasContent) : ?>
                        <div class="text-sm"><?= $content ?></div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php

        return trim(ob_get_clean());
    }
}
